﻿using Performance.DtoModels;
using Performance.EntityModels;
using System;
using System.Collections.Generic;
using System.Linq;

namespace Performance.Repository
{
    public partial class PerforReportRepository : PerforRepository<PerReport>
    {
        public PerforReportRepository(PerformanceDbContext context) : base(context)
        {
        }

        /// <summary>
        /// 月群体人均绩效
        /// </summary>
        /// <returns></returns>
        public List<PerReport> GetAvgPerfor(int hospitalid)
        {

            string sql = @"select concat(allot.year,'-',lpad(allot.month,2,'0')) x, positionname y,round(avgvalue,2) value
                from res_baiscnorm bc left join per_allot allot on bc.allotid = allot.id where allot.hospitalid = @hospitalid 
                order by str_to_date(concat(allot.month, '/', allot.year),'%m/%Y')";
            return DapperQuery(sql, new { hospitalid }).ToList();
        }

        /// <summary>
        /// 人群绩效比
        /// </summary>
        /// <param name="hospitalid"></param>
        /// <returns></returns>
        public List<PerReport> AvgRatio(int hospitalid)
        {
            string sql = @"select concat(allot.year,'-',lpad(allot.month,2,'0')) x, bc.PositionName y,round(bc.AvgValue / rbn.AvgValue,2) value
                from res_baiscnorm bc left join res_baiscnorm rbn on bc.allotid = rbn.allotid and rbn.positionname = '临床科室主任人均绩效'
                left join per_allot allot on bc.allotid = allot.id where allot.hospitalid = @hospitalid 
                order by str_to_date(concat(allot.month, '/', allot.year),'%m/%Y');";
            return DapperQuery(sql, new { hospitalid }).ToList();
        }

        /// <summary>
        /// 门诊患者均次费用
        /// </summary>
        /// <returns></returns>
        public List<PerReport> OutFeeAvg(int hospitalId, List<string> date)
        {
            string sql = @"select concat(year,'-',lpad(month,2,'0')) y,deptname x,round(sum(fee) / sum(persontime),2) value from hos_personfee where source = '门诊' and concat(year,'-',lpad(month,2,'0')) and hospitalid = @hospitalId in @date group by year,month,deptname order by y asc,value desc;";
            return DapperQuery(sql, new { date, hospitalId }).ToList();
        }

        /// <summary>
        /// 住院患者均次费用
        /// </summary>
        /// <returns></returns>
        public List<PerReport> InpatFeeAvg(int hospitalId, List<string> date)
        {
            string sql = @"select concat(year,'-',lpad(month,2,'0')) y,deptname x,round(sum(fee) / sum(persontime),2) value from hos_personfee where source = '住院' and concat(year,'-',lpad(month,2,'0')) in @date and hospitalid = @hospitalId group by year,month,deptname order by y asc,value desc;";
            return DapperQuery(sql, new { date, hospitalId }).ToList();
        }

        /// <summary>
        /// 科室药占比
        /// </summary>
        /// <returns></returns>
        public List<PerReport> Medicine(int hospitalId, List<string> date)
        {
            string sql = @"select accountingunit x,concat(year,'-',lpad(month,2,'0')) y,round((sum(if(cd.id is null,0,cellvalue))  / sum(cellvalue))*100,2) value 
                from per_allot aot join per_sheet sht on aot.id=sht.allotid join im_data dt on dt.sheetid=sht.id 
				left join cof_drugtype cd on cd.allotid=dt.allotid and cd.charge=dt.TypeName and cd.chargetype in ('药费') where unittype=1 and sheettype=3 
                and sheetname like '%开单收入' and ifnull(accountingunit,'') not in ('') and concat(year,'-',lpad(month,2,'0')) 
                in @date and hospitalid = @hospitalId group by year,month,accountingunit order by y asc,value desc;";
            return DapperQuery(sql, new { hospitalId, date }).ToList();
        }

        /// <summary>
        /// 科室有效收入占比
        /// </summary>
        /// <returns></returns>
        public List<PerReport> Income(int hospitalId, List<string> date)
        {
            string sql = @"select accountingunit x,concat(year,'-',lpad(month,2,'0')) y,round((sum(if(cd.id is null,cellvalue,0)) / sum(cellvalue))*100,2) value
                from per_allot aot join per_sheet sht on aot.id=sht.allotid join im_data dt on dt.sheetid=sht.id 
				left join cof_drugtype cd on cd.allotid=dt.allotid and cd.charge=dt.TypeName and cd.chargetype in ('药费','材料费') 
				where unittype=1 and sheettype=3 and sheetname like '%开单收入' and ifnull(accountingunit,'') not in ('') and concat(year,'-',lpad(month,2,'0')) 
                in @date and hospitalid = @hospitalId group by year,month,accountingunit order by y asc,value desc;";
            return DapperQuery(sql, new { hospitalId, date }).ToList();
        }

        #region 首页报表
        /// <summary>
        /// 医院收入结构占比
        /// </summary>
        /// <returns></returns>
        public List<PerReport> InHosIncome(int hospitalId, string currentDate)
        {
            string sql = $"select '{currentDate}' x,'住院收入占比' y,round(t.InHos/(t.InHos+t.Outpatient) * 100, 2) value from (select year,month,sum(case SourceType when '住院' then CellValue else 0 end) InHos,sum(case SourceType when '门诊' then CellValue else 0 end) Outpatient from view_report_original_income where hospitalid = @hospitalId and concat(year,'-',lpad(month,2,'0')) = '{currentDate}' group by year,month)t" +
              $" union " +
              $"select '{currentDate}' x,'门诊收入占比' y,round(t.Outpatient/(t.InHos+t.Outpatient) * 100, 2) value from (select year,month,sum(case SourceType when '住院' then CellValue else 0 end) InHos,sum(case SourceType when '门诊' then CellValue else 0 end) Outpatient from view_report_original_income where hospitalid = @hospitalId and concat(year,'-',lpad(month,2,'0')) = '{currentDate}' group by year,month)t;";
            return DapperQuery(sql, new { hospitalId }).ToList();
        }

        /// <summary>
        /// 绩效发放金额
        /// </summary>
        /// <param name="currentDate"></param>
        /// <param name="yoyDate"></param>
        /// <param name="chainDate"></param>
        /// <returns></returns>
        public List<PerReport> PerforPayment(int hospitalId, string currentDate, string yoyDate, string chainDate)
        {
            string sql = $"select '{currentDate}' x,t1.y,ifnull(t2.RealGiveFee, 0.0000) value from (select '{currentDate}' x,'当月发放金额' y union select '{yoyDate}' x,'同期发放金额' y union select '{chainDate}' x,'环比发放金额' y)t1 left join view_report_allot_summary t2 on t1.x = concat(t2.year,'-',lpad(t2.month,2,'0')) and t2.HospitalId = @hospitalId order by y;";
            return DapperQuery(sql, new { hospitalId }).ToList();
        }

        /// <summary>
        /// 绩效发放金额占全院收入占比
        /// </summary>
        /// <param name="currentDate"></param>
        /// <param name="yoyDate"></param>
        /// <param name="chainDate"></param>
        /// <returns></returns>
        public List<PerReport> IndexPerforRatio(int hospitalId, string currentDate, string yoyDate, string chainDate)
        {
            string sql = $"select '{currentDate}' x,t1.y, round(ifnull(t2.RealGiveFee, 0.0000)/t3.income*100, 2)value from(select '{currentDate}' x,'当月发放占比' y union select '{yoyDate}' x,'同期发放占比' y union select '{chainDate}' x,'环比发放占比' y)t1 left join view_report_allot_summary t2 on t1.x = concat(t2.year,'-', lpad(t2.month,2,'0')) and t2.HospitalId = @hospitalId left join (select year, month, sum(CellValue) income from view_report_original_income where HospitalId = @hospitalId group by year, month)t3 on t1.x = concat(t3.year,'-', lpad(t3.month,2,'0')) order by y;";
            return DapperQuery(sql, new { hospitalId }).ToList();
        }

        /// <summary>
        /// 首页药占比（本月、环比、同期）
        /// </summary>
        /// <param name="hospitalId">医院Id</param>
        /// <param name="currentDate">本月日期</param>
        /// <param name="yoyDate">同比日期</param>
        /// <param name="chainDate">环比日期</param>
        /// <returns></returns>
        public List<PerReport> IndexDrugRatio(int hospitalId, string currentDate, string yoyDate, string chainDate)
        {
            string sql = $"select '{currentDate}' x,t1.y,round(t2.CellValue/t3.`value`*100, 2) value from (select '{currentDate}' x,'当月药占比' y union select '{yoyDate}' x,'同期药占比' y union select '{chainDate}' x,'环比药占比' y)t1 left join (select concat(year,'-',lpad(month,2,'0')) date,sum(CellValue) CellValue from view_report_original_income t1 inner join cof_drugtype t2 on t1.TypeName = t2.Charge and t1.AllotID = t2.AllotId and t2.ChargeType = '药费' where HospitalID = @hospitalId group by year,month) t2 on t1.x = t2.date left join (select concat(year,'-',lpad(month,2,'0')) date,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId group by year,month)t3 on t1.x = t3.date order by y;";
            return DapperQuery(sql, new { hospitalId }).ToList();
        }

        /// <summary>
        /// 首页材料占比（本月、环比、同期）
        /// </summary>
        /// <param name="hospitalId">医院Id</param>
        /// <param name="currentDate">本月日期</param>
        /// <param name="yoyDate">同比日期</param>
        /// <param name="chainDate">环比日期</param>
        /// <returns></returns>
        public List<PerReport> IndexMaterialRatio(int hospitalId, string currentDate, string yoyDate, string chainDate)
        {
            string sql = $"select '{currentDate}' x,t1.y,round(t2.CellValue/t3.`value`*100, 2) value from (select '{currentDate}' x,'当月材料占比' y union select '{yoyDate}' x,'同期材料占比' y union select '{chainDate}' x,'环比材料占比' y)t1 left join (select concat(year,'-',lpad(month,2,'0')) date,sum(CellValue) CellValue from view_report_original_income t1 inner join cof_drugtype t2 on t1.TypeName = t2.Charge and t1.AllotID = t2.AllotId and t2.ChargeType = '材料费' where HospitalID = @hospitalId group by year,month) t2 on t1.x = t2.date left join (select concat(year,'-',lpad(month,2,'0')) date,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId group by year,month)t3 on t1.x = t3.date order by y;";
            return DapperQuery(sql, new { hospitalId }).ToList();
        }

        /// <summary>
        /// 首页结构占比
        /// </summary>
        /// <returns></returns>
        public List<PerReport> IndexStructRatio(int hospitalId, string currentDate)
        {
            string sql = $"select TypeName y,t1.y x,round(CellValue/t2.`value`*100,2) value from (select '{currentDate}' y,TypeName,sum(CellValue) CellValue from view_report_original_income where HospitalID = @hospitalId and concat(year,'-',lpad(month,2,'0')) = '{currentDate}' group by TypeName) t1 inner join (select '{currentDate}' y,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId and concat(year,'-',lpad(month,2,'0')) = '{currentDate}')t2 on t1.y = t2.y order by value desc;";
            return DapperQuery(sql, new { hospitalId }).ToList();
        }
        #endregion

        #region 菜单报表
        /// <summary>
        /// 业务总收入
        /// </summary>
        /// <returns></returns>
        public List<PerReport> GeneralIncome(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and month in ({request.Month}) ";
            }
            string sql = $"select concat(year,'年') x,'业务总收入（年）' y,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId {where} group by year order by x;";
            if (request.OnlyYear != 1)
                sql = $"select concat(month,'月') x,concat(year,'年') y,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId {where} group by year,month order by x;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }

        /// <summary>
        /// 门诊、住院业务收入占比
        /// </summary>
        /// <returns></returns>
        public List<PerReport> InHosIncome(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and month in ({request.Month}) ";
            }
            string sql = $"select * from (select '住院' y,concat(year,'年') x,round(t.InHos/(t.InHos + t.Outpatient) * 100, 2) value from (select year,sum(case SourceType when '住院' then CellValue else 0 end) InHos,sum(case SourceType when '门诊' then CellValue else 0 end) Outpatient from view_report_original_income where hospitalid = @hospitalId {where} group by year)t" +
                $" union " +
                $"select '门诊' y,concat(year,'年') x,round(t.Outpatient/(t.InHos + t.Outpatient) * 100, 2) value from (select year,sum(case SourceType when '住院' then CellValue else 0 end) InHos,sum(case SourceType when '门诊' then CellValue else 0 end) Outpatient from view_report_original_income where hospitalid = @hospitalId {where} group by year)t)t order by t.x;";
            if (request.OnlyYear != 1)
                sql = $"select * from (select '住院' y,concat(month,'月') x,round(t.InHos/(t.InHos+t.Outpatient) * 100, 2) value from (select year,month,sum(case SourceType when '住院' then CellValue else 0 end) InHos,sum(case SourceType when '门诊' then CellValue else 0 end) Outpatient from view_report_original_income where hospitalid = @hospitalId {where} group by month)t" +
                    $" union " +
                    $"select '门诊' y,concat(month,'月') x,round(t.Outpatient/(t.InHos+t.Outpatient) * 100, 2) value from (select year,month,sum(case SourceType when '住院' then CellValue else 0 end) InHos,sum(case SourceType when '门诊' then CellValue else 0 end) Outpatient from view_report_original_income where hospitalid = @hospitalId {where} group by month)t)t order by t.x;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }

        /// <summary>
        /// 业务收入结构占比
        /// </summary>
        /// <returns></returns>
        public List<PerReport> StructRatio(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and month in ({request.Month}) ";
            }
            string sql = $"select TypeName y,concat(t1.y,'年') x,round(CellValue/t2.`value`*100,2) value from (select year y,TypeName,sum(CellValue) CellValue from view_report_original_income where HospitalID = @hospitalId {where} group by year,TypeName) t1 inner join (select year y,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId {where} group by year)t2 on t1.y = t2.y order by x asc,value desc;";
            if (request.OnlyYear != 1)
                sql = $"select TypeName y,concat(t1.y,'月') x,round(CellValue/t2.`value`*100,2) value from (select month y,TypeName,sum(CellValue) CellValue from view_report_original_income where HospitalID = @hospitalId {where} group by month,TypeName) t1 inner join (select month y,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId {where} group by month)t2 on t1.y = t2.y order by x asc,value desc;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }

        /// <summary>
        /// 药占比
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="request.OnlyYear"></param>
        /// <returns></returns>
        public List<PerReport> DrugRatio(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and month in ({request.Month}) ";
            }
            string sql = $"select '药占比（年）' y,concat(t1.y,'年') x,round(CellValue/t2.`value`*100,2) value from (select year y,sum(CellValue) CellValue from view_report_original_income t1 inner join cof_drugtype t2 on t1.TypeName = t2.Charge and t1.AllotID = t2.AllotId and t2.ChargeType = '药费' where HospitalID = @hospitalId {where} group by year) t1 inner join (select year y,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId {where} group by year)t2 on t1.y = t2.y order by x asc;";
            if (request.OnlyYear != 1)
                sql = $"select concat(t1.year,'年') y,concat(t1.month,'月') x,round(CellValue/t2.`value`*100,2) value from (select month,year,sum(CellValue) CellValue from view_report_original_income t1 inner join cof_drugtype t2 on t1.TypeName = t2.Charge and t1.AllotID = t2.AllotId and t2.ChargeType = '药费' where HospitalID = @hospitalId {where} group by month,year) t1 inner join (select month,year,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId {where} group by month,year)t2 on t1.`month`=t2.`month` and t1.year = t2.year order by x asc;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }

        /// <summary>
        /// 材料占比
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="request.OnlyYear"></param>
        /// <returns></returns>
        public List<PerReport> MaterialRatio(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and month in ({request.Month}) ";
            }
            string sql = $"select '材料占比（年）' y,concat(t1.y,'年') x,round(CellValue/t2.`value`*100,2) value from (select year y,sum(CellValue) CellValue from view_report_original_income t1 inner join cof_drugtype t2 on t1.TypeName = t2.Charge and t1.AllotID = t2.AllotId and t2.ChargeType = '材料费' where HospitalID = @hospitalId {where} group by year) t1 inner join (select year y,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId {where} group by year)t2 on t1.y = t2.y order by x asc;";
            if (request.OnlyYear != 1)
                sql = $"select concat(t1.year,'年') y,concat(t1.month,'月') x,round(CellValue/t2.`value`*100,2) value from (select month,year,sum(CellValue) CellValue from view_report_original_income t1 inner join cof_drugtype t2 on t1.TypeName = t2.Charge and t1.AllotID = t2.AllotId and t2.ChargeType = '材料费' where HospitalID = @hospitalId {where} group by month,year) t1 inner join (select month,year,sum(CellValue) value from view_report_original_income where hospitalid = @hospitalId {where} group by month,year)t2 on t1.`month`=t2.`month` and t1.year = t2.year order by x asc;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }

        /// <summary>
        /// 绩效发放金额占全院收入占比
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="request.OnlyYear"></param>
        /// <returns></returns>
        public List<PerReport> PerforRatio(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and month in ({request.Month}) ";
            }
            string sql = $"select concat(t2.x,'年') x,'绩效发放金额占全院收入占比（年）' y,round(t1.RealGiveFee/t2.income * 100, 2) value from (select `year`,sum(realgivefee) realgivefee from view_report_allot_summary where HospitalID = @hospitalId {where} group by year) t1 inner join (select year x,sum(CellValue) income from view_report_original_income where HospitalId = @hospitalId {where} group by year)t2 on t1.`Year` = t2.x order by x asc;";
            if (request.OnlyYear != 1)
                sql = $"select concat(t2.month,'月') x,concat(t2.year,'年') y,round(t1.RealGiveFee/t2.income * 100, 2) value from (select `year`,`month`,sum(realgivefee) realgivefee from view_report_allot_summary where HospitalID = @hospitalId {where} group by year,month) t1 inner join (select `year`,`month`,sum(CellValue) income from view_report_original_income where HospitalId = @hospitalId {where} group by year,month)t2 on t1.month = t2.month and t1.year = t2.year order by x asc;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }

        /// <summary>
        /// 绩效群体收入
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="request.OnlyYear"></param>
        /// <returns></returns>
        public List<PerReport> PerforGroup(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and month in ({request.Month}) ";
            }
            string sql = $"select PositionName y,concat(year,'年') x,round(sum(avgvalue), 2) value from (select t1.PositionName,year,AvgValue from res_baiscnorm t1 inner join per_allot t2 on t1.AllotID = t2.Id and t2.states in (6,8,10) and t2.HospitalId = @hospitalId {where} and locate('保底', t1.PositionName) = 0)t group by PositionName,year order by x asc,value desc;";
            if (request.OnlyYear != 1)
                sql = $"select PositionName y,concat(month,'月') x,round(sum(avgvalue), 2) value from (select t1.PositionName,month,AvgValue from res_baiscnorm t1 inner join per_allot t2 on t1.AllotID = t2.Id and t2.states in (6,8,10) and t2.HospitalId = @hospitalId {where} and locate('保底', t1.PositionName) = 0)t group by PositionName,month order by x asc,value desc;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }

        /// <summary>
        /// 医生核算单元人均绩效
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="request.OnlyYear"></param>
        /// <returns></returns>
        public List<PerReport> DoctorAvg(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and t2.year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and t2.month in ({request.Month}) ";
            }
            string sql = $"select concat(`Year`,'年') x,AccountingUnit y,round(sum(avg),2) value from (select t1.AccountingUnit,t1.Avg,t2.`Year`,t2.`Month`,t2.ID from res_account t1 inner join per_allot t2 on t1.AllotID = t2.ID and t2.states in (6,8,10) and t2.HospitalId = @hospitalId and t1.UnitType != 2 where 1=1 {where})t group by year,AccountingUnit order by x asc,value desc;";
            if (request.OnlyYear != 1)
                sql = $"select concat(`Month`,'月') x,AccountingUnit y,round(sum(avg),2) value from (select t1.AccountingUnit,t1.Avg,t2.`Year`,t2.`Month`,t2.ID from res_account t1 inner join per_allot t2 on t1.AllotID = t2.ID and t2.states in (6,8,10) and t2.HospitalId = @hospitalId and t1.UnitType != 2 where 1=1 {where})t group by month,AccountingUnit order by x asc,value desc;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }

        /// <summary>
        /// 护理核算单元人均绩效
        /// </summary>
        /// <param name="hospitalId"></param>
        /// <param name="request.OnlyYear"></param>
        /// <returns></returns>
        public List<PerReport> NurseAvg(ReportRequest request)
        {
            string where = "";
            if (!string.IsNullOrEmpty(request.Year))
            {
                where += $" and t2.year in ({request.Year}) ";
            }
            if (!string.IsNullOrEmpty(request.Month))
            {
                where += $" and t2.month in ({request.Month}) ";
            }
            string sql = $"select concat(`Year`,'年') x,AccountingUnit y,round(sum(avg),2) value from (select t1.AccountingUnit,t1.Avg,t2.`Year`,t2.`Month`,t2.ID from res_account t1 inner join per_allot t2 on t1.AllotID = t2.ID and t2.states in (6,8,10) and t2.HospitalId = @hospitalId and t1.UnitType = 2 where 1=1 {where})t group by year,AccountingUnit order by x asc,value desc;";
            if (request.OnlyYear != 1)
                sql = $"select concat(`Month`,'月') x,AccountingUnit y,round(sum(avg),2) value from (select t1.AccountingUnit,t1.Avg,t2.`Year`,t2.`Month`,t2.ID from res_account t1 inner join per_allot t2 on t1.AllotID = t2.ID and t2.states in (6,8,10) and t2.HospitalId = @hospitalId and t1.UnitType = 2 where 1=1 {where})t group by month,AccountingUnit order by x asc,value desc;";
            return DapperQuery(sql, new { hospitalId = request.HospitalId }).ToList();
        }
        #endregion
    }
}
